#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Professional cPanel Installer for VPN Bot & WebApp
This script automates the deployment process on cPanel Python Hosting.
"""

import os
import sys
import time
import subprocess
import getpass
import shutil
from datetime import datetime

# ANSI Colors for Professional Output
class Colors:
    HEADER = '\033[95m'
    BLUE = '\033[94m'
    CYAN = '\033[96m'
    GREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'

def print_banner():
    print(f"""{Colors.CYAN}
    ================================================================
       VPN BOT & WEBAPP - PROFESSIONAL CPANEL INSTALLER
    ================================================================
    {Colors.ENDC}""")

def print_step(step, message):
    print(f"{Colors.BLUE}[STEP {step}]{Colors.ENDC} {Colors.BOLD}{message}{Colors.ENDC}")

def print_success(message):
    print(f"{Colors.GREEN}✔ {message}{Colors.ENDC}")

def print_error(message):
    print(f"{Colors.FAIL}✖ {message}{Colors.ENDC}")

def print_info(message):
    print(f"{Colors.CYAN}ℹ {message}{Colors.ENDC}")

def check_python_version():
    print_step(1, "Checking Environment...")
    if sys.version_info < (3, 8):
        print_error("Python 3.8 or higher is required.")
        sys.exit(1)
    print_success(f"Python Version: {sys.version.split()[0]}")

def get_user_input():
    print_step(2, "Configuration Setup")
    print_info("Please provide the following information to configure your bot.")
    
    config = {}
    
    # Bot Configuration
    print(f"\n{Colors.HEADER}--- Telegram Bot Settings ---{Colors.ENDC}")
    while True:
        config['BOT_TOKEN'] = input(f"{Colors.BOLD}Enter Bot Token:{Colors.ENDC} ").strip()
        if config['BOT_TOKEN']: break
        print_error("Bot Token is required!")

    while True:
        try:
            config['ADMIN_ID'] = int(input(f"{Colors.BOLD}Enter Admin Telegram ID:{Colors.ENDC} ").strip())
            break
        except ValueError:
            print_error("Please enter a valid number for Admin ID.")

    config['BOT_USERNAME'] = input(f"{Colors.BOLD}Enter Bot Username (without @):{Colors.ENDC} ").strip()
    
    # Channel Configuration
    print(f"\n{Colors.HEADER}--- Channel Settings (Optional - Press Enter to skip) ---{Colors.ENDC}")
    config['CHANNEL_ID'] = input(f"{Colors.BOLD}Enter Channel ID (e.g., @MyChannel):{Colors.ENDC} ").strip()
    config['CHANNEL_LINK'] = input(f"{Colors.BOLD}Enter Channel Link (e.g., https://t.me/MyChannel):{Colors.ENDC} ").strip()
    config['REPORTS_CHANNEL_ID'] = input(f"{Colors.BOLD}Enter Reports Channel ID (Numeric):{Colors.ENDC} ").strip()
    config['RECEIPTS_CHANNEL_ID'] = input(f"{Colors.BOLD}Enter Receipts Channel ID (Numeric):{Colors.ENDC} ").strip()
    
    # Database Configuration
    print(f"\n{Colors.HEADER}--- Database Settings (MySQL) ---{Colors.ENDC}")
    print_info("You must create a MySQL database and user in cPanel first.")
    
    config['MYSQL_HOST'] = input(f"{Colors.BOLD}MySQL Host [localhost]:{Colors.ENDC} ").strip() or 'localhost'
    config['MYSQL_PORT'] = input(f"{Colors.BOLD}MySQL Port [3306]:{Colors.ENDC} ").strip() or '3306'
    
    while True:
        config['MYSQL_DATABASE'] = input(f"{Colors.BOLD}Database Name:{Colors.ENDC} ").strip()
        if config['MYSQL_DATABASE']: break
        print_error("Database Name is required!")
        
    while True:
        config['MYSQL_USER'] = input(f"{Colors.BOLD}Database User:{Colors.ENDC} ").strip()
        if config['MYSQL_USER']: break
        print_error("Database User is required!")
        
    while True:
        config['MYSQL_PASSWORD'] = getpass.getpass(f"{Colors.BOLD}Database Password:{Colors.ENDC} ").strip()
        if config['MYSQL_PASSWORD']: break
        print_error("Database Password is required!")

    # WebApp Configuration
    print(f"\n{Colors.HEADER}--- WebApp Settings ---{Colors.ENDC}")
    config['WEBAPP_URL'] = input(f"{Colors.BOLD}Enter Your Domain URL (e.g., https://mydomain.com):{Colors.ENDC} ").strip()
    if config['WEBAPP_URL'] and not config['WEBAPP_URL'].startswith('http'):
        config['WEBAPP_URL'] = 'https://' + config['WEBAPP_URL']

    return config

def create_env_file(config):
    print_step(3, "Generating Configuration File (.env)...")
    
    env_content = f"""# VPN Bot Configuration - Generated by cPanel Installer
# Date: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}

# Telegram Bot
BOT_TOKEN={config['BOT_TOKEN']}
ADMIN_ID={config['ADMIN_ID']}
BOT_USERNAME={config['BOT_USERNAME']}

# Channels
# Channels
CHANNEL_ID={config.get('CHANNEL_ID', '')}
CHANNEL_LINK={config.get('CHANNEL_LINK', '')}
REPORTS_CHANNEL_ID={config.get('REPORTS_CHANNEL_ID', '0')}
RECEIPTS_CHANNEL_ID={config.get('RECEIPTS_CHANNEL_ID', '0')} 

# WebApp
WEBAPP_URL={config.get('WEBAPP_URL', '')}
BOT_WEBAPP_URL={config.get('WEBAPP_URL', '')}
USE_HTTPS=true

# Database
MYSQL_HOST={config['MYSQL_HOST']}
MYSQL_PORT={config['MYSQL_PORT']}
MYSQL_DATABASE={config['MYSQL_DATABASE']}
MYSQL_USER={config['MYSQL_USER']}
MYSQL_PASSWORD={config['MYSQL_PASSWORD']}

# Security
FLASK_SECRET_KEY={os.urandom(24).hex()}
"""
    
    try:
        with open('.env', 'w', encoding='utf-8') as f:
            f.write(env_content)
        print_success(".env file created successfully.")
    except Exception as e:
        print_error(f"Failed to create .env file: {e}")
        sys.exit(1)

def install_requirements():
    print_step(4, "Installing Dependencies...")
    print_info("This may take a few minutes. Please wait...")
    
    try:
        subprocess.check_call([sys.executable, "-m", "pip", "install", "-r", "requirements.txt"])
        print_success("Dependencies installed successfully.")
    except subprocess.CalledProcessError:
        print_error("Failed to install dependencies. Please check the error output above.")
        sys.exit(1)

def create_passenger_wsgi():
    print_step(5, "Configuring Web Server (passenger_wsgi.py)...")
    
    wsgi_content = """import sys
import os

# Add current directory to sys.path
sys.path.insert(0, os.path.dirname(__file__))

# Import the Flask app
from webapp import app as application
"""
    
    try:
        # Backup existing file if it exists
        if os.path.exists('passenger_wsgi.py'):
            shutil.copy('passenger_wsgi.py', 'passenger_wsgi.py.bak')
            print_info("Backed up existing passenger_wsgi.py to passenger_wsgi.py.bak")
            
        with open('passenger_wsgi.py', 'w', encoding='utf-8') as f:
            f.write(wsgi_content)
        print_success("passenger_wsgi.py created successfully.")
    except Exception as e:
        print_error(f"Failed to create passenger_wsgi.py: {e}")

def initialize_database():
    print_step(6, "Initializing Database...")
    
    try:
        # We need to load environment variables first since we just created .env
        from dotenv import load_dotenv
        load_dotenv()
        
        from professional_database import ProfessionalDatabaseManager
        db = ProfessionalDatabaseManager()
        
        # Test connection
        with db.get_connection() as conn:
            print_success("Database connection successful!")
            
        # The ProfessionalDatabaseManager initializes tables in __init__
        # But we can explicitly check/create if needed. 
        # For now, just connecting confirms it's working and init logic ran.
        print_success("Database tables initialized.")
        
    except Exception as e:
        print_error(f"Database initialization failed: {e}")
        print_info("Please check your database credentials in .env file.")

def create_start_script():
    print_step(7, "Creating Startup Script (start_bot.sh)...")
    
    # Get python path and current directory
    python_path = sys.executable
    current_dir = os.getcwd()
    
    script_content = f"""#!/bin/bash

# Navigate to project directory
cd {current_dir}

# Check if bot is running
if pgrep -f "telegram_bot.py" > /dev/null
then
    echo "Bot is already running."
    exit 0
fi

# Start bot
echo "Starting bot..."
{python_path} telegram_bot.py >> bot.log 2>&1 &
"""
    
    try:
        with open('start_bot.sh', 'w', encoding='utf-8') as f:
            f.write(script_content)
        
        # Make executable
        os.chmod('start_bot.sh', 0o755)
        print_success("start_bot.sh created and made executable.")
    except Exception as e:
        print_error(f"Failed to create start_bot.sh: {e}")

def print_final_instructions(config):
    print(f"""
    {Colors.GREEN}================================================================
       INSTALLATION COMPLETED SUCCESSFULLY!
    ================================================================{Colors.ENDC}
    
    {Colors.BOLD}Next Steps:{Colors.ENDC}
    
    1. {Colors.CYAN}Restart Python App:{Colors.ENDC}
       Go to cPanel > Setup Python App > Restart your application.
       
    2. {Colors.CYAN}Set up Telegram Bot Cron Job:{Colors.ENDC}
       The Telegram bot needs to run continuously. We have created a script 'start_bot.sh' for this.
       
       Go to cPanel > Cron Jobs
       Add a new Cron Job to run every minute (* * * * *):
       
       {Colors.WARNING}/bin/bash {os.getcwd()}/start_bot.sh{Colors.ENDC}
       
    3. {Colors.CYAN}Access your WebApp:{Colors.ENDC}
       Open: {config.get('WEBAPP_URL', 'Your Domain')}
       
    {Colors.BLUE}Thank you for using this Professional Installer!{Colors.ENDC}
    """)

def main():
    try:
        print_banner()
        check_python_version()
        config = get_user_input()
        create_env_file(config)
        install_requirements()
        create_passenger_wsgi()
        initialize_database()
        create_start_script()
        print_final_instructions(config)
    except KeyboardInterrupt:
        print(f"\n\n{Colors.WARNING}Installation cancelled by user.{Colors.ENDC}")
        sys.exit(0)
    except Exception as e:
        print_error(f"An unexpected error occurred: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    main()
